## This script takes results from the base scenario in step 5 and produces charts and values used in the report

rm(list=ls())
gc()

## The results from ALL scenarios cannot be pulled in at once due to size limits.
## So this script must be run with a chosen set of results corresponding to a category of scenario analysis adjustments (see appendix)
## Sets 1.1 and 2.1 are used to produce outputs presented in the chapter
## Sets 1, 2, 3, 4 are used to produce outputs presented in the appendix

## Pick which set of results to source:
## set 1.1: rate of return adjustments covered in chapter
## set 1.2: other rate of return adjustments (not used on their own)
## set 1: 1.1 and 1.2 (for appendix charts)
## set 2.1: wealth transfer adjustments covered in chapter
## set 2.2: other wealth transfer adjustments (not used on their own)
## set 2: 2.1 and 2.2 (for appendix charts)
## set 3: housing adjustments
## set 4: other adjustments
set <- 4


# Prelims -----------------------------------------------------------------

memory.limit(120000)

## ensure packages are loaded
source("./R scripts/Master package loading.R")

## source extra data and functions for analysing results
source("./R scripts/Simulation model/Cohort simulation results - functions prep.R")


## version of results to read in, by date
run_date <- "2021-10-26" 


# Read in and tidy results data ----------------------------------------------------

## restrict results 2018, 2048, and every five years from 2020 to make dataframes more manageable
restricted_years <- c(1, 31, seq(3, 33, 5)) ## this is according to order of list
restricted_years_gini_only <- c(seq(3, 33, 5)) ## this covers years required for the Gini by generation plots only

## base case
sim_results_w_transfers <- qread(paste0("./OUtput data/sim_results_returns_converge_", run_date, ".qs")) %>% 
  .[restricted_years] %>% 
  tidy_results_fn %>% 
  mutate(model = "Base scenario")

sim_results_no_transfers <- qread(paste0("./OUtput data/sim_results_returns_converge_no_transfer_", run_date, ".qs")) %>% 
  .[restricted_years] %>% 
  tidy_results_fn %>% 
  mutate(model = "Base scenario no transfers")



## RATE OF RETURN ADJUSTMENTS -------

if(set %in% c(1.1, 1)) {
  ## 1a: Historical rates
  sim_results_historic <- qread(paste0("./Output data/sim_results_historic_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "1a: Historical rates")
  
  sim_results_historic_no_transfers <- qread(paste0("./Output data/sim_results_historic_no_transfer_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "1a: Historical rates no transfers")
  

  ## 1b: High returns
  sim_results_high_return <- qread(paste0("./Output data/sim_results_high_growth_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "1b: High returns")
  
  sim_results_high_return_no_transfers <- qread(paste0("./Output data/sim_results_high_growth_no_transfer_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "1b: High returns no transfers")
  
  
  ## 1c: Low returns
  sim_results_low_return <- qread(paste0("./Output data/sim_results_low_growth_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "1c: Low returns")
  
  sim_results_low_return_no_transfers <- qread(paste0("./Output data/sim_results_low_growth_no_transfer_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "1c: Low returns no transfers")
  
  
  ## 1d: Zero returns
  sim_results_zero_returns <- qread(paste0("./Output data/sim_results_zero_returns_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "1d: Zero returns")
  
  sim_results_zero_returns_no_transfers <- qread(paste0("./Output data/sim_results_zero_returns_no_transfer_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "1d: Zero returns no transfers")
  
}

if(set %in% c(1.2, 1)) {
  ## 1e: Housing return 5%
  sim_results_housing_cf <- qread(paste0("./Output data/sim_results_housing_cf_", run_date, ".qs")) %>%
    .[restricted_years] %>%
    tidy_results_fn %>%
    mutate(model = "1e: Housing return 5%")
  
  sim_results_housing_cf_no_transfers <- qread(paste0("./Output data/sim_results_housing_cf_no_transfer_", run_date, ".qs")) %>%
    .[restricted_years] %>%
    tidy_results_fn %>%
    mutate(model = "1e: Housing return 5% no transfers")

}

## WEALTH TRANSFER ADJUSTMENTS ------
if(set %in% c(2.1, 2))  {
  
  ## 2a: Wealthy receive larger inheritances
  sim_results_low_wealth_low_beq_dist <- qread(paste0("./Output data/sim_results_low_wealth_low_beq_dist_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "2a: Wealthy receive larger inheritances")
  
  sim_results_low_wealth_low_beq_dist_no_transfers <- qread(paste0("./Output data/sim_results_low_wealth_low_beq_dist_no_transfer_", run_date, ".qs")) %>% 
    .[restricted_years_gini_only] %>% 
    tidy_results_fn %>% 
    mutate(model = "2a: Wealthy receive larger inheritances no transfers")
  
  ## 2b: Wealthy save more of inheritance
  sim_results_diff_bequest_saving <- qread(paste0("./Output data/sim_results_low_wealth_beq_depletion_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "2b: Wealthy save more of inheritance")
  
  sim_results_diff_bequest_saving_no_transfers <- qread(paste0("./Output data/sim_results_low_wealth_beq_depletion_no_transfer_", run_date, ".qs")) %>% 
    .[restricted_years_gini_only] %>% 
    tidy_results_fn %>% 
    mutate(model = "2b: Wealthy save more of inheritance no transfers")
  
  ## 2c: Less wealthy save less before inherit
  sim_results_low_wealth_low_saving_b4 <- qread(paste0("./Output data/sim_results_low_wealth_low_saving_b4_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "2c: Less wealthy save less before inherit   ")
  
  sim_results_low_wealth_low_saving_b4_no_transfers <- qread(paste0("./Output data/sim_results_low_wealth_low_saving_b4_no_transfer_", run_date, ".qs")) %>% 
    .[restricted_years_gini_only] %>% 
    tidy_results_fn %>% 
    mutate(model = "2c: Less wealthy save less before inherit    no transfers")
  
}

if(set %in% c(2.1)) {
  ## for "fan" charts in chapter
  
  ## Housing return 5%
  sim_results_housing_cf <- qread(paste0("./Output data/sim_results_housing_cf_", run_date, ".qs")) %>% 
    .[restricted_years_gini_only] %>% 
    tidy_results_fn %>% 
    mutate(model = "Housing return 5%")
  
  ## Housing return 3%
  sim_results_housing_cf2 <- qread(paste0("./Output data/sim_results_housing_cf2_", run_date, ".qs")) %>% 
    .[restricted_years_gini_only] %>% 
    tidy_results_fn %>% 
    mutate(model = "Housing return 3%")
}


if(set %in% c(2.2, 2))  {
  ## 2d: Older save more of inheritance
  sim_results_increase_bequest_saving <- qread(paste0("./Output data/sim_results_increase_bequest_saving_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "2d: Older save more of inheritance")
  
  sim_results_increase_bequest_saving_no_transfers <- qread(paste0("./Output data/sim_results_increase_bequest_saving_no_transfer_", run_date, ".qs")) %>% 
    .[restricted_years_gini_only] %>% 
    tidy_results_fn %>% 
    mutate(model = "2d: Older save more of inheritance no transfers")
  
  
  ## 2e: Save 80% inheritance
  sim_results_bequest_saving_all <- qread(paste0("./Output data/sim_results_bequest_saving_most_", run_date, ".qs")) %>%
    .[restricted_years] %>%
    tidy_results_fn %>%
    mutate(model = "2e: Save 80% inheritance")

  sim_results_bequest_saving_all_no_transfers <- qread(paste0("./Output data/sim_results_bequest_saving_most_no_transfer_", run_date, ".qs")) %>%
    .[restricted_years_gini_only] %>%
    tidy_results_fn %>%
    mutate(model = "2e: Save 80% inheritance no transfers")
  

  ## 2f: Housing for gifts
  sim_results_housing_for_gifts <- qread(paste0("./Output data/sim_results_housing_for_gifts_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "2f: Housing for gifts")
  
  sim_results_housing_for_gifts_no_transfers <- qread(paste0("./Output data/sim_results_housing_for_gifts_no_transfer_", run_date, ".qs")) %>% 
    .[restricted_years_gini_only] %>% 
    tidy_results_fn %>% 
    mutate(model = "2f: Housing for gifts no transfers")
  
  
}
 

## HOUSING ADJUSTMENTS ------
if(set==3) {
  ## 3a: Alt method for home transitions
  sim_results_alt_home_prob <- qread(paste0("./Output data/sim_results_alt_home_prob_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "3a: Alt method for home transitions")
  
  sim_results_alt_home_prob_no_transfers <- qread(paste0("./Output data/sim_results_alt_home_prob_no_transfer_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "3a: Alt method for home transitions no transfers")
  
  
  ## 3b: Housing debt to asset ratio 83%
  sim_results_housing_debt83 <- qread(paste0("./Output data/sim_results_housing_debt83_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "3b: Housing debt to asset ratio 83%")
  
  sim_results_housing_debt83_no_transfers <- qread(paste0("./Output data/sim_results_housing_debt83_no_transfer_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "3b: Housing debt to asset ratio 83% no transfers")
  
  
  ## 3c: Same housing entry prices
  sim_results_same_housing_entry_price <- qread(paste0("./Output data/sim_results_same_housing_entry_price_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "3c: Same housing entry prices")
  
  sim_results_same_housing_entry_price_no_transfers <- qread(paste0("./Output data/sim_results_same_housing_entry_price_no_transfer_", run_date, ".qs")) %>% 
    .[restricted_years_gini_only] %>% 
    tidy_results_fn %>% 
    mutate(model = "3c: Same housing entry prices no transfers")
  
  
  ## 3d: Housing entry prices increase
  sim_results_higher_housing_entry_price <- qread(paste0("./Output data/sim_results_higher_housing_entry_price_", run_date, ".qs")) %>%
    .[restricted_years] %>%
    tidy_results_fn %>%
    mutate(model = "3d: Housing entry prices increase")

  sim_results_higher_housing_entry_price_no_transfers <- qread(paste0("./Output data/sim_results_higher_housing_entry_price_no_transfer_", run_date, ".qs")) %>%
    .[restricted_years] %>%
    tidy_results_fn %>%
    mutate(model = "3d: Housing entry prices increase no transfers")
  

  ## 3e: Housing affordability declines
  sim_results_house_afford <- qread(paste0("./Output data/sim_results_declining_house_afford_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "3e: Housing affordability declines")
  
  sim_results_house_afford_no_transfers <- qread(paste0("./Output data/sim_results_declining_house_afford_no_transfer_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "3e: Housing affordability declines no transfers")
  
  
  ## 3f: Lower housing demand
  sim_results_lower_housing_demand <- qread(paste0("./Output data/sim_results_lower_housing_demand_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "3f: Lower housing demand")
  
  sim_results_lower_housing_demand_no_transfers <- qread(paste0("./Output data/sim_results_lower_housing_demand_no_transfer_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "3f: Lower housing demand no transfers")

}


## OTHER ADJUSTMENTS --------------------------
if(set==4) {
  ## 4a: Alt method for other asset drawdowns
  sim_results_alt_other_drawdown <- qread(paste0("./Output data/sim_results_alt_other_drawdown_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "4a: Alt method for other asset drawdowns")
  
  sim_results_alt_other_drawdown_no_transfers <- qread(paste0("./Output data/sim_results_alt_other_drawdown_no_transfer_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "4a: Alt method for other asset drawdowns no transfers")
  
  
  ## 4b: Older drawdowns increase
  sim_results_increase_drawdowns_larger <- qread(paste0("./Output data/sim_results_increase_drawdowns_larger_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "4b: Older drawdowns increase")
  
  sim_results_increase_drawdowns_larger_no_transfers <- qread(paste0("./Output data/sim_results_increase_drawdowns_larger_no_transfer_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "4b: Older drawdowns increase no transfers")
  
  
  ## 4c: All wealth to younger gens at death
  sim_results_all_wealth_passed <- qread(paste0("./Output data/sim_results_all_wealth_passed_", run_date , ".qs")) %>% #"2021-09-13"
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "4c: All wealth to younger gens at death")
  
  sim_results_all_wealth_passed_no_transfers <- qread(paste0("./Output data/sim_results_all_wealth_passed_no_transfer_", run_date, ".qs")) %>% 
    .[restricted_years] %>% 
    tidy_results_fn %>% 
    mutate(model = "4c: All wealth to younger gens at death no transfers")
  
  
}


# Combined results data -------------------------------------------

## This will combine data from the sets that have been read in
combined_scenario_results <- rbindlist(
  ls() %>% str_subset("sim_results") %>% lapply(., get),
  fill=T
)

## remove to free up memory
rm(list= ls() %>% str_subset("sim_results") )

## Chart namer for appendix charts
if(set==1 ) {chart_id <- "set1" ; scenario_order <- c("Base scenario", "1a: Historical rates", "1b: High returns", "1c: Low returns", "1d: Zero returns",  "1e: Housing return 5%") }
if(set==2 ) {chart_id <- "set2" ; scenario_order <- c("Base scenario", "2a: Wealthy receive larger inheritances", "2b: Wealthy save more of inheritance", "2c: Less wealthy save less before inherit   ", 
                                                     "2d: Older save more of inheritance", "2e: Save 80% inheritance", "2f: Housing for gifts") }
if(set==3 ) {chart_id <- "set3" ; scenario_order <- c("Base scenario", "3a: Alt method for home transitions", "3b: Housing debt to asset ratio 83%", "3c: Same housing entry prices", "3d: Housing entry prices increase", "3e: Housing affordability declines", "3f: Lower housing demand")}
if(set==4 ) {chart_id <- "set4" ; scenario_order <- c("Base scenario", "4a: Alt method for other asset drawdowns", "4b: Older drawdowns increase", "4c: All wealth to younger gens at death")}



# Summarised dataframes by group -------------------------------------------------------


## by year and age
combined_results_year_age <- combined_scenario_results %>% 
  group_by(model, year, age_grp) %>% 
  summarise_vars_fn 

## Gini by generation (for plots later)
gini_gen <- combined_scenario_results %>% 
  ## filter to millenials and gen x
  filter(year %in% seq(2020, 2050, 5) &
           !(gen0 %in% c("Post-millennials" , "Boomers" , "Pre-boomers")) &
           age_grp<="[95,100)") %>% 
  group_by(model, year, gen0) %>% 
  group_split %>% 
  ## for each year and generation, calc the Gini coefficient for that pop
  lapply(., function(x) {
    data.table(
      model = x[[1, "model"]],
      year=x[[1, "year"]],
      gen0= x[[1, "gen0"]],
      gini=gini(x$total_wealth, weights=x$n)
    )
  }) %>% 
  rbindlist %>% 
  mutate(transfer_status = ifelse(str_detect(model, "no transfers"), "gini_no_transfers", "gini_w_transfers"),
         model = str_replace(model, " no transfers", "")) %>% 
  pivot_wider(names_from = transfer_status, values_from=gini) %>% 
  ## calculate diff
  mutate(gini_diff = gini_no_transfers-gini_w_transfers)



# CHAPTER PLOTS AND VALUES -----------------------------------------------------------

# Set 1.1 calcs --------------------------------------

if(set==1.1) {
  ## Size of total wealth for 60-65
  ## *** USED IN CHAPTER
  ## Compared with the base scenario where all rates of return converge to 4 per cent over ten years, estimates of average wealth held by people aged 60–64 in 2048 were projected to be:
  # •	65 per cent lower in the zero returns scenario, where future wealth is accumulated wholly through savings and transfers
  # •	30 per cent lower in the low returns scenario, where rates converge to 2.4 per cent
  # •	44 per cent higher in the high returns scenario, where rates converge to 5.6 per cent.
  ## Average wealth held by people aged 60–64 was 88 per cent higher than in the base scenario (in the historical rates case)
  combined_results_year_age %>% 
    ungroup %>% 
    filter(age_grp=="[65,70)" & year==2048 & !str_detect(model, "no transfers")) %>% 
    select(model, total_wealth_tot_real, total_wealth_av_real) %>% 
    mutate(base_share = total_wealth_tot_real/total_wealth_tot_real[5]-1)
  
  ## Historical returns scenario. How much in housing?
  ## *** USED IN CHAPTER
  ##  People aged 30–99 held 73 per cent of their total wealth in housing in 2048, relative to 61 per cent under the base scenario. 
  housing_all <- combined_results_year_age %>% 
    filter(model %in% c("Base scenario", "1a: Historical rates") & age_grp >= "[30,35)" & age_grp <= "[95,100)" & year == 2048) %>% 
    mutate(housing_wealth = housing_assets_tot_real + housing_debt_tot_real) %>% 
    #filter(older==1) %>% 
    group_by(model, year) %>% 
    summarise( ## percentage of own wealth held in housing / in super
      housing_age_pc = sum(housing_wealth) / sum(total_wealth_tot_real),
      super_age_pc = sum(super_assets_tot_real) / sum(total_wealth_tot_real))
  housing_all
  
  
  ## chart of total wealth by age in 2048 - figure 3.7a
  asset_growth_total_wealth_age_line <- ggplot(combined_results_year_age %>% 
                                                 filter(year==2048 & !str_detect(model, "no transfers")) %>% 
                                                 mutate(model = factor(model, levels=c("Base scenario", c("1d: Zero returns", "1c: Low returns", "1b: High returns", "1a: Historical rates") %>% rev) ))
  ) +
    geom_line(aes(x=as.numeric(age_grp), y=total_wealth_tot_real/billion, colour=model)) +
    scale_x_continuous(labels = unique(combined_results_year_age$age_grp)[seq(1, 21, 3)] %>% age_grp_labeller,
                       breaks = seq(1, 21, 3)) +
    scale_y_continuous(expand=c(0,0), limits = c(0, 3500)) +
    ylab("Total wealth ($b)") +
    xlab("Age group") +
    scale_colour_pc() +
    custom_plot_margin
  
  emf(file=paste0("./Charts/set1-1_tot_wealth_age_2048", Sys.Date(), ".emf"), 
      width = 7.5/2.54, height = 6/2.54,
      pointsize=12,
      family="Arial")
  asset_growth_total_wealth_age_line  +
    theme(legend.position = "none")
  dev.off()
  
  ggsave(file=paste0("./Charts/set1-1_tot_wealth_age_2048", Sys.Date(), ".svg"), 
         plot=  asset_growth_total_wealth_age_line  +
           theme(legend.position = "none"), 
         width = 7.5/2.54, height = 6/2.54)
         
  legend <-   asset_growth_total_wealth_age_line +
    coord_cartesian(xlim=c(0,1), ylim=c(0,1)) +
    theme(panel.grid = element_blank(),
          axis.line = element_blank(),
          axis.title.x = element_blank(),
          axis.title.y = element_blank(),
          axis.text.x = element_blank(),
          axis.text.y = element_blank(),
          axis.ticks = element_blank()) +
    guides(colour=guide_legend(nrow=2))
  
  ## wide legend only
  emf(file=paste0("./Charts/set1-1_rates_legend", Sys.Date(), ".emf"), 
      width = 15/2.54, height = 1.5/2.54,
      pointsize=12,
      family="Arial")
  legend
  dev.off()
  
  ggsave(file=paste0("./Charts/set1-1_rates_legend", Sys.Date(), ".svg"), 
         plot=legend, 
         width = 15/2.54, height = 1.5/2.54)
  
  
  # temp <- combined_results_year_age %>% 
  #   filter(year==2048 & !str_detect(model, "no transfers")) %>% 
  #   mutate(model = factor(model, levels=c("Base scenario", c("1d: Zero returns", "1c: Low returns", "1b: High returns", "1a: Historical rates") %>% rev) )) %>% 
  #   select(model, age_grp, total_wealth_tot_real) %>% 
  #   mutate(total_wealth_tot_real=round(total_wealth_tot_real/billion, 3), 
  #          age_grp=age_grp_labeller(age_grp))
  # write.csv(temp, "temp.csv")
  
  ## gini difference for millenials -  figure 3.7b
  asset_growth_gini_diff_gen_plot <- ggplot(gini_gen %>% 
                                              filter(gen0=="Millennials") %>% 
                                              mutate(model = factor(model, levels=c("Base scenario", c("1d: Zero returns", "1c: Low returns", "1b: High returns", "1a: Historical rates") %>% rev) ))) +
    geom_line(aes(x=year, y=gini_diff*100, colour=model)) +
    scale_y_continuous(limits=c(0, 0.9), expand=c(0,0)) +
    ylab("Reduction in Gini coefficient") +
    xlab("Year") +
    scale_colour_pc() +
    custom_plot_margin +
    ## Arrows and labels
    geom_segment(aes(x=2022, xend=2022, y=0.65, yend=0.8),
                 colour= pc_grey,
                 size=0.05,
                 arrow=arrow(length = unit(0.2, "cm"), type="closed")
    ) +
    geom_text(aes(label = "More equal", x=2023.5, y=0.725), size=2.9, hjust=0)
  
  emf(file=paste0("./Charts/set1-1_gini_millenials", Sys.Date(), ".emf"), 
      width = 7.5/2.54, height = 6/2.54,
      pointsize=12,
      family="Arial")
  asset_growth_gini_diff_gen_plot  +
    theme(legend.position = "none")
  dev.off()
  
  ggsave(file=paste0("./Charts/set1-1_gini_millenials", Sys.Date(), ".svg"), 
         plot=  asset_growth_gini_diff_gen_plot  +
           theme(legend.position = "none"), 
         width = 7.5/2.54, height = 6/2.54)
  
  
  # temp <- gini_gen %>%
  #   filter(gen0=="Millennials") %>%
  #   mutate(model = factor(model, levels=c("Base scenario", c("1d: Zero returns", "1c: Low returns", "1b: High returns", "1a: Historical rates") %>% rev) )) %>%
  #   select(model, year, gini_diff) %>%
  #   mutate(gini_diff=round(gini_diff*100, 3))
  # write.csv(temp, "temp.csv")
}



# Set 2.1 calcs ------------------------------------------------------

if(set==2.1) {
  
  ## calc diff in Gini between the housing cfs and the base scenario
    gini_gen_housing_cf <- combined_scenario_results %>% 
      ## filter to 
      filter(model %in% c("Base scenario", "Housing return 5%", "Housing return 3%") & 
               year %in% seq(2020, 2050, 5) &
               !(gen0 %in% c("Post-millennials" , "Boomers" , "Pre-boomers")) &
               age_grp<="[95,100)") %>% 
      group_by(model, year, gen0) %>% 
      group_split %>% 
      ## for each year, calc the Gini coefficient for that pop
      lapply(., function(x) {
        #print(x[[1, "model_year"]])
        data.table(
          model = x[[1, "model"]],
          year=x[[1, "year"]],
          gen0= x[[1, "gen0"]],
          gini=gini(x$total_wealth, weights=x$n)
        )
      }) %>% 
      rbindlist %>% 
      group_by(year, gen0) %>% 
      mutate(gini_base = gini[1],
             gini_diff = gini - gini_base) %>% 
      filter(model!="Base scenario")
    
    
  ## create the chart - figure 3.8
  gini_diff_gen_plot <- ggplot(gini_gen %>% 
                                 filter(!str_detect(model, "Housing") ) %>% 
                                 mutate(model = factor(model, levels=c("Base scenario", "2a: Wealthy receive larger inheritances", "2b: Wealthy save more of inheritance", "2c: Less wealthy save less before inherit   ")))
  )  +
    geom_area(data = gini_gen_housing_cf,
              aes(x = year, y=gini_diff*100, fill=model), alpha=0.5) +
    geom_hline(yintercept=0, colour=pc_grey, linetype=1, size=0.2) +
    geom_line(aes(x=year, y=gini_diff*100, colour=model)) +
    facet_wrap(vars(gen0), scales = "free_x") +
    ylab("Reduction in Gini coefficient") +
    xlab("Year") +
    guides(colour=guide_legend(ncol=2),
           fill=F) +
    theme(panel.spacing = unit(2, "lines")) +
    scale_colour_pc() +
    scale_fill_manual(values=c(pc_grey, pc_grey)) +
    custom_plot_margin +
    ## Arrows and labels
    geom_segment(aes(x=2022, xend=2022, y=0.8, yend=1.1),
                 colour= pc_grey,
                 size=0.05,
                 arrow=arrow(length = unit(0.2, "cm"), type="closed")
    ) +
    geom_segment(aes(x=2022, xend=2022, y=-0.75, yend=-1.05),
                 colour= pc_grey,
                 size=0.05,
                 arrow=arrow(length = unit(0.2, "cm"), type="closed")
    ) +
    geom_text(aes(label = "More equal", x=2023.5, y=1), size=2.9, hjust=0) +
    geom_text(aes(label = "Less equal", x=2023.5, y=-0.95), size=2.9, hjust=0) +
    theme(legend.justification = c(0,1),
          legend.margin = margin(t = 0, r = 0, b = 0, l = -15, unit = "pt"))
  
  emf(file=paste0("./Charts/set2-1_gini_gen", Sys.Date(), ".emf"), 
      width = 15/2.54, height = 10/2.54,
      pointsize=12,
      family="Arial")
  gini_diff_gen_plot 
  dev.off()
  
  ggsave(file=paste0("./Charts/set2-1_gini_gen", Sys.Date(), ".svg"), 
         plot=gini_diff_gen_plot, 
         width = 15/2.54, height = 10/2.54)
  
  
  
  # temp <- rbind(
  #   gini_gen %>% 
  #     filter(!str_detect(model, "Housing") ) %>% 
  #     mutate(model = factor(model, levels=c("Base scenario", "2a: Wealthy receive larger inheritances", "2b: Wealthy save more of inheritance", "2c: Less wealthy save less before inherit   "))) %>% 
  #     select(model, gen0, year, gini_diff) %>%
  #     arrange(model, gen0, year, gini_diff) %>% 
  #     data.table,
  #   gini_gen_housing_cf %>% 
  #     select(model, gen0, year, gini_diff) %>% 
  #     arrange(model, gen0, year, gini_diff) %>% 
  #     data.table
  # ) %>% 
  #   mutate(gini_diff=round(gini_diff*100, 3))
  # write.csv(temp, "temp.csv")


}



# APPENDIX PLOTS AND VALUES - figures C.8-C.11 ----------------------------------------------------------


# Panel a: Charts of wealth by age at 2048 ------------------------------------------

if(set==1) {chart_limits <- c(0,3500)}
if(set==2) {chart_limits <- c(0,2000)}
if(set==3) {chart_limits <- c(0,3500)}
if(set==4) {chart_limits <- c(0,2000)}

## *** PLOT USED IN APPENDIX
total_wealth_age_2050_line <- ggplot(combined_results_year_age %>% 
                                       filter(year==2048 & !str_detect(model, "no transfers") ) %>% 
                                       mutate(model = factor(model,
                                                             levels= scenario_order),
                                              base = ifelse(model=="Base scenario", 1, 0)) )  +
  geom_line(aes(x=as.numeric(age_grp), y=total_wealth_tot_real/billion, 
                colour=model)) +
  geom_line(data= combined_results_year_age %>% 
              filter(year==2048 & model=="Base scenario") ,
            aes(x=as.numeric(age_grp), y=total_wealth_tot_real/billion ), ## base scenario always on top, dashed. It'll show up if too much overlap
            colour=pc_blue, linetype="dashed"
            ) + 
  scale_x_continuous(labels = unique(combined_results_year_age$age_grp)[seq(1, 21, 3)] %>% age_grp_labeller,
                     breaks = seq(1, 21, 3)) +
  scale_y_continuous(expand=c(0,0), limits=chart_limits) +
  ylab("Total wealth ($b)") +
  xlab("Age group") +
  guides(colour=guide_legend(ncol=3),
         linetype=F) +
  scale_colour_manual(values = c(pc_pal, pc_grey) ) +
  custom_plot_margin

emf(file=paste0("./Charts/", chart_id, "_total_wealth", Sys.Date(), ".emf"), 
    width = 7.5/2.54, height = 6/2.54,
    pointsize=12,
    family="Arial")
total_wealth_age_2050_line +
  theme(legend.position = "none")
dev.off()

ggsave(file=paste0("./Charts/", chart_id, "_total_wealth", Sys.Date(), ".svg"), 
       plot=total_wealth_age_2050_line +
         theme(legend.position = "none"), 
       width = 7.5/2.54, height = 6/2.54)


# temp <- combined_results_year_age %>%
#   filter(year==2048 & !str_detect(model, "no transfers") ) %>%
#   mutate(model = factor(model,
#                         levels= scenario_order),
#          base = ifelse(model=="Base scenario", 1, 0)) %>%
#   select(model, age_grp, total_wealth_tot_real) %>%
#   arrange(model, age_grp, total_wealth_tot_real) %>%
#   mutate(total_wealth_tot_real = round(total_wealth_tot_real/billion, 3),
#          age_grp=age_grp_labeller(age_grp))
# write.csv(temp, "temp.csv")


# Panel b: Charts of shares of wealth held by age group in 2048------------------------------------------

## TEXT IN APPENDIX FOR 4c
## 4c: as a percentage of the population aged 30–99, there was a 19 percentage point increase in the wealth share 
## compared with a 12 percentage point increase in population share for people who were 60 and over
if(set==4) {
  intergen_dist_over60 <- combined_results_year_age %>% 
    mutate(model = ifelse(year==2018 & model=="Base scenario", "Base scenario 2018", model)) %>% 
    filter( age_grp>="[30,35)" & age_grp<="[95,100)") %>% 
    filter((year==2048  & str_detect(model, "Base scenario|4c") & !str_detect(model, "no transfers")) | model=="Base scenario 2018") %>% 
    mutate(over = ifelse(age_grp>="[60,65)", 1, 0)) %>% 
    group_by(model) %>% 
    mutate(total_wealth_year = sum(total_wealth_tot_real ),
           n_year = sum(n_tot)) %>% 
    ungroup %>% 
    select(model, over, total_wealth_tot_real,n_tot, total_wealth_year, n_year  ) %>% 
    group_by(model, over) %>% 
    summarise(age_pc = sum(total_wealth_tot_real)/mean(total_wealth_year),
              n_pc = sum(n_tot)/ mean(n_year)) %>% 
    select(model, over, age_pc, n_pc)
  
  ## increase in wealth share
  data.table(intergen_dist_over60)[model=="4c: All wealth to younger gens at death" & over==1, "age_pc"] - data.table(intergen_dist_over60)[model=="Base scenario 2018" & over==1, "age_pc"]
  
  ## increase in pop share
  data.table(intergen_dist_over60)[model=="4c: All wealth to younger gens at death" & over==1, "n_pc"] - data.table(intergen_dist_over60)[model=="Base scenario 2018" & over==1, "n_pc"]
}


## At 2048 - the distribution of wealth between young and old? 
## (Note there can be very slight diffs between n if the scenario results in changes to home/beqrec transition probabilities leading to removal of some pathways)
intergen_dist <- combined_results_year_age %>% 
  filter((year==2048 ) & age_grp>="[30,35)" & age_grp<="[95,100)") %>% 
  group_by(model) %>% 
  mutate(total_wealth_year = sum(total_wealth_tot_real ),
         n_year = sum(n_tot)) %>% 
  ungroup %>% 
  mutate(age_pc = (total_wealth_tot_real)/total_wealth_year,
         n_pc = n_tot/n_year) %>% 
  select(model, age_grp, age_pc, n_pc)


## plot *** USED IN APPENDIX
intergen_dist_plot <- ggplot(intergen_dist %>% 
                               filter(!str_detect(model, "no transfers")) %>% 
                               mutate(model = factor(model, 
                                                     levels= scenario_order) )) +  
  geom_col(data = intergen_dist %>% filter(model=="Base scenario") %>% mutate(model = "Population %"),
           aes(x=as.numeric(age_grp), y=n_pc*100, fill=model)) +
  geom_line(aes(x=as.numeric(age_grp), y=age_pc*100, colour=model)) +
  geom_line(data = intergen_dist %>% filter(model=="Base scenario"),
            aes(x=as.numeric(age_grp), y=age_pc*100),
            colour = pc_blue, linetype="dashed"
  ) + ## dashed base scenario always on top for visibility
  scale_x_continuous(labels = unique(intergen_dist$age_grp)[seq(1, 14, 3)] %>% age_grp_labeller(),
                     breaks = seq(7, 20, 3), 
                     expand=c(0,0)) +
  scale_y_continuous(expand=c(0,0), limits=c(0,15)) +
  ylab("Per cent") +
  xlab("Age group") +
  guides(colour=guide_legend(ncol=3)) +
  scale_colour_manual(values=c(pc_pal, pc_grey)) +
  scale_fill_manual(values=pc_light_grey) +
  custom_plot_margin

emf(file=paste0("./Charts/", chart_id, "_intergen_wealth", Sys.Date(), ".emf"), 
    width = 7.5/2.54, height = 6/2.54,
    pointsize=12,
    family="Arial")
intergen_dist_plot +
  theme(legend.position = "none")
dev.off()

ggsave(file=paste0("./Charts/", chart_id, "_intergen_wealth", Sys.Date(), ".svg"), 
       plot=intergen_dist_plot +
         theme(legend.position = "none"), 
       width = 7.5/2.54, height = 6/2.54)


legend <- intergen_dist_plot +
  coord_cartesian(xlim=c(0,1), ylim=c(0,1)) +
  guides(colour=guide_legend(ncol=2, order=1)) +
  guides(fill=guide_legend(order=2)) +
  theme(panel.grid = element_blank(),
        axis.line = element_blank(),
        axis.title.x = element_blank(),
        axis.title.y = element_blank(),
        axis.text.x = element_blank(),
        axis.text.y = element_blank(),
        axis.ticks = element_blank()) +
  theme(legend.spacing.x = unit(0.5, 'cm'))  +
  theme(legend.box="vertical",
        legend.box.just = "left",
        legend.spacing.y = unit(0, "cm"))

## legend
emf(file=paste0("./Charts/", chart_id, "_app_legend", Sys.Date(), ".emf"), 
    width = 15/2.54, height = 3/2.54,
    pointsize=12,
    family="Arial")
legend
dev.off()

ggsave(file=paste0("./Charts/", chart_id, "_app_legend", Sys.Date(), ".svg"), 
       plot=legend, 
       width = 15/2.54, height = 3/2.54)

# temp <- intergen_dist %>%
#   filter(!str_detect(model, "no transfers")) %>%
#   mutate(model = factor(model,
#                         levels= scenario_order),
#          age_grp = age_grp_labeller(age_grp),
#          across(c(age_pc, n_pc), ~round(.x*100,2)) )  %>%
#   arrange(model, age_grp)
# write.csv(temp, "temp.csv")

  
# Panel c: Charts of relative inequality by gen ----------------------------------------------

## plot scenarios
# gini_gen_plot <- ggplot(gini_gen %>% filter(gen0!="Post-millennials")) +
#   geom_line(aes(x=year, y=gini_w_transfers*100, colour=model)) +
#   facet_wrap(vars(gen0)) +
#   guides(colour=guide_legend(ncol=3))

if(set==1) {chart_limits <- c(0,0.9)}
if(set==2) {chart_limits <- c(-0.75,1.2)}
if(set==3) {chart_limits <- c(0,1.2)}
if(set==4) {chart_limits <- c(0,0.75)}

## *** USED IN APPENDIX
gini_diff_gen_plot <- ggplot(gini_gen %>% 
                               mutate(model = factor(model,
                                                        levels = scenario_order))) +
  geom_hline(yintercept=0, colour=pc_grey, linetype=1, size=0.2) +
  geom_line(aes(x=year, y=gini_diff*100, colour=model)) +
  geom_line(data = gini_gen %>% filter(model=="Base scenario"), 
            aes(x=year, y=gini_diff*100),
            colour=pc_blue, linetype="dashed") + ## base scenario dashed line always on top so visible with overlap
  facet_wrap(vars(gen0)) +
  scale_y_continuous(expand=c(0,0), limits=chart_limits) +
  ylab("Reduction in Gini coefficient") +
  xlab("Year") +
  guides(colour=guide_legend(ncol=3)) +
  scale_colour_manual(values=c(pc_pal, pc_grey)) +
  custom_plot_margin +
  theme(panel.spacing = unit(2, "lines")) +
  ## Arrows and labels
  geom_segment(aes(x=2022, xend=2022, y=chart_limits[2]-0.2, yend=chart_limits[2]-0.05),
               colour= pc_grey,
               size=0.05,
               arrow=arrow(length = unit(0.2, "cm"), type="closed")
  ) +
  geom_segment(aes(x=2022, xend=2022, y=-0.5, yend=-0.65),
               colour= pc_grey,
               size=0.05,
               arrow=arrow(length = unit(0.2, "cm"), type="closed")
  ) +
  geom_text(aes(label = "More equal", x=2023.5, y=chart_limits[2]-0.1), size=2.9, hjust=0) +
  geom_text(aes(label = "Less equal", x=2023.5, y=-0.6), size=2.9, hjust=0) +
  theme(legend.justification = c(0,1),
        legend.margin = margin(t = 0, r = 0, b = 0, l = -15, unit = "pt"))

emf(file=paste0("./Charts/", chart_id, "_gini_gen_diff", Sys.Date(), ".emf"), 
    width = 15/2.54, height = 7/2.54,
    pointsize=12,
    family="Arial")
gini_diff_gen_plot +
  theme(legend.position = "none")
dev.off()

ggsave(file=paste0("./Charts/", chart_id, "_gini_gen_diff", Sys.Date(), ".svg"), 
       plot=gini_diff_gen_plot +
         theme(legend.position = "none"), 
       width = 15/2.54, height = 7/2.54)


# temp <- gini_gen %>%
#   mutate(model = factor(model,
#                         levels = scenario_order)) %>%
#   select(model, gen0, year, gini_diff) %>%
#   mutate(gini_diff = round(gini_diff*100, 3)) %>%
#   arrange(model, gen0, year)
# write.csv(temp, "temp.csv")



# Misc set 1 - Gini projections in appendix text -----------------------------------------

if(set %in% c(1.1, 1)) {
  
  ## Gini
  gc()
  gini <- combined_scenario_results[year %in% c(2018, 2048)   & age_grp>="[30,35)" & age_grp<="[95,100)" & model %in% c("Base scenario", "1a: Historical rates"),
                                    c("model", "year", "n", "total_wealth")]
  ## filter age groups as well as some disappear over time
  ## filter to age group because no births mean no younger ages at later years. 100-105 2020 sensitive 
  gini[, model_year := paste0(model, year)]
  gini <- split(gini, gini$model_year) %>% 
    ## for each year, calc the Gini coefficient for that pop
    lapply(., function(x) {
      print(x[[1, "model_year"]])
      data.table(
        model = x[[1, "model"]],
        year=x[[1, "year"]],
        gini=gini(x$total_wealth, weights=x$n)
      )
    }) %>% 
    rbindlist 
  
  ## USED IN APPENDIX
  ## Projections of the Gini coefficient for the 30–99 age group in the base scenario indicated a fall in inequality by 
  ## about 5 percentage points between 2018 and 2048. These projections were sensitive to rate of return assumptions 
  ## — for example, when historical rates of return on assets were assumed in the model, the Gini was projected to be 
  ## little changed in 2048. 
  gini
}

